/*===========================================================================



LW TX driver unit control firmware
==================================



Overview:
=========

CPU         :   PIC16F84A
Clock       :   XT crystal oscillator, 4MHz
Compiler    :   Hitech PICC
Version     :   1.02
Date        :   16.08.2011
Author      :   HB9DUL



I/O connections:
================

RA0  ---->  TX_RELAY    Transmit relay (1=transmit)
RA1  ---->  ON_AIR_LED  On air LED (1=on)
RA2  ---->  TX_SEND     MOSFET enable line (1=transmit)
RA3  ---->  ALARM_LED   Alarm LED (1=on)
RA4  -----              Not connected

RB0  <----  SW_STBY     Stand by / transmit switch (0=transmit)
RB1  <----  SW_BKIN     Break-in / manual switch (0=break-in)
RB2  <----  CW_KEY      CW key connection (0=closed)
RB3  <----  ALARM_1     Alarm 1 input line (1=alarm)
RB4  <----  ALARM_2     Alarm 2 input line (1=alarm)
RB5  -----              Not connected
RB6  -----              Not connected
RB7  -----              Not connected



Description:
============

This program controls the transmitting functions of the longwaves 
transmitter.
It's main function is to drive the antenna relay and RF power MOSFETs,
making sure that a transmission is only possible when the relay has fully
switched the antenna with the transmitter output. 
It also handles the alarms immediately switching off the transmitter if 
an alarm condition is detected.
At startup, an initial delay allows all voltages to set before starting
the transmitter, than a safety check is done preventing transmission if 
the "standby" switch was forgotten in the "transmit" position.
The software is basically composed by a state machine summarized below:

                  ###################
                  # STATE_STARTUP   #
                  ###################
                           
                           
                           V
 +--------------- ################### <------------------------------+
                 # STATE_STBY      #                                
           +---- ################### <-----+                        
                                                                  
                                                                  
           V                                                       
    ###################        ###################                  
    # STATE_SWTX      # -----> # STATE_SWRX      #                  
    ###################        ###################                  
                               A             A                    
                                                                
                                                                
               ###################         ###################    
          +---> # STATE_TX        # <-----> # STATE_TX_ON_AIR #    
                ###################         ###################    
                                                                 
       +----------+                   +-------+                    
                                                                 
                  V       V            V                            
           #####################################                    
           # STATE_ALARM_1   # STATE_ALARM_2   #--------------------+
 +--------> #####################################

A detailed description of every state is available in the code below.

============================================================================= */



// --------------------------------------------------------------------------
// External files inclusions.
// --------------------------------------------------------------------------
#include <pic.h>                                                // Use a PIC16F84A.



// --------------------------------------------------------------------------
// Configuration bits definitions.
// --------------------------------------------------------------------------
__CONFIG(XT & WDTDIS & PWRTEN & UNPROTECT);
// Use internal XT crystal oscillator, do not use watchdog, enable power
// up timer and do not use data protection.



// --------------------------------------------------------------------------
// Definitions, constants and global variables.
// --------------------------------------------------------------------------

// I/O pins.
#define TX_RELAY        RA0     // Transmit relay (1=transmit)
#define ON_AIR_LED      RA1     // On air LED (1=on)
#define TX_SEND         RA2     // MOSFET enable line (1=transmit)
#define ALARM_LED       RA3     // Alarm LED (1=on)

#define SW_STBY         RB0     // Stand by / transmit switch (0=transmit)
#define SW_BKIN         RB1     // Break-in / manual switch (0=break-in)
#define CW_KEY          RB2     // CW key connection (0=closed)
#define ALARM_1         RB3     // Alarm 1 input line (1=alarm)
#define ALARM_2         RB4     // Alarm 2 input line (1=alarm)

// Signal polarities.
#define     SW_STBY_STBY    1
#define     SW_STBY_TX      0
#define     SW_BKIN_MAN     1
#define     SW_BKIN_BKIN    0
#define     CW_KEY_UP       1
#define     CW_KEY_DOWN     0
#define     ALARM_ON        1
#define     ALARM_OFF       0
#define     TX_RELAY_ON     1
#define     TX_RELAY_OFF    0
#define     TX_SEND_ON      1
#define     TX_SEND_OFF     0
#define     LED_ON          1
#define     LED_OFF         0

// State equivalents.
#define     STATE_STARTUP   0
#define     STATE_STBY      10
#define     STATE_SWTX      20
#define     STATE_TX        21
#define     STATE_TX_ON_AIR 22
#define     STATE_SWRX      23
#define     STATE_ALARM_1   30
#define     STATE_ALARM_2   31

// Timing. (time is expressed in number of 1.024ms periods ( = (4 * 4 * 256) / 4MHz))
#define     RELAY_ON_DELAY  12                                  // 12.3ms for the relay to switch on.
#define     RELAY_OFF_DELAY 12                                  // 12.3ms for the relay to switch off.
#define     LED_BLINK_TIME  244                                 // 249.9ms as LED blinking half period (~2Hz)
#define     STARTUP_TIME    488                                 // 0.5s to wait at startup to allow all voltages to set.
#define     BREAK_IN_TIME   977                                 // 1s to wait before tripping the TX relay back.



// --------------------------------------------------------------------------
// Global variables.
// --------------------------------------------------------------------------
unsigned char state;                                            // Current state of the state machine.
unsigned int counter;                                           // Counter incremented by TMR0 interrupt: used to measure time.



// --------------------------------------------------------------------------
// This procedure is the interrupt handler. It's called by TMR0, which is 
// set up to generate an interrupt at a frequency of 976.6 Hz 
// ( = 4MHz / (4 * 4 * 256) ) or an interrupty every 1.024ms.
// --------------------------------------------------------------------------
void interrupt interrupt_handler(void)
{
    GIE = 0;                                                    // Disable interrupts.
    if (T0IF == 1)                                              // TMR0 overflow?
    {
        counter++;                                              // Increment counter.
        T0IF = 0;                                               // Clear TMR0 interrupt flag.
    }
    GIE = 1;                                                    // Enable interrupts.
}



// --------------------------------------------------------------------------
// This is the main program.
// --------------------------------------------------------------------------
void main(void)
{

// --------------------------------------------------------------------------
// I/O lines initialization.
// --------------------------------------------------------------------------
    PORTA = 0b00000000;                                         // Reset PORTA.
//                
//                +-- --> TX_RELAY
//                +--- --> ON_AIR_LED
//                +---- --> TX_SEND
//                +----- --> ALARM_LED
    TRISA = 0b00000000;                                         // PORTA all outputs.

    PORTB = 0b00000000;                                         // Reset PORTB.
//               
//               +-- <-- SW_STBY
//               +--- <-- SW_BKIN
//               +---- <-- CW_KEY
//               +----- <-- ALARM_1
//               +------ <-- ALARM_2
    RBPU = 0;                                                   // Enable weak pullup on PORTB.
    TRISB = 0b11111111;                                         // PORTB all inputs.

// --------------------------------------------------------------------------
// TMR0 and its interrupt initialization.
// --------------------------------------------------------------------------
    TMR0 = 0;                                                   // Clear TMR0.
    T0CS = 0;                                                   // Selct internal clock for TMR0.
    PSA = 0;                                                    // Assign prescaler to TMR0.
    PS2 = 0; PS1 = 0; PS0 = 1;                                  // Set prescaler to 1:4 (TMR0 clock at 250kHz = 4MHz / (4 * 4) ).
    T0IF = 0;                                                   // Clear interrupt flag.
    T0IE = 1;                                                   // Enable TMR0 interrupt.

    GIE = 1;                                                    // Enable interrupts.

// --------------------------------------------------------------------------
// Main loop (state machine).
// --------------------------------------------------------------------------
    state = STATE_STARTUP;                                      // Start in standby mode.
    while (1)                                                   // The main loop.
    {
        switch (state)
        {

// --------------------------------------------------------------------------
// In the "startup" state, first the predefined delay in the STARTUP_TIME
// constant is waited in order to make sure all voltages are set and steady
// before starting the transmitter, than a safety check is performed to
// prevent the transmitter from going on air when powered up by monitoring
// the "standby" switch. If it has been forgotten in the "transmit" position,
// the "on air" LED will blink until the switch is put back in the "standby"
// position, preventing all transmission. When the status of this switch has
// verified, the state changes to "standby".
// --------------------------------------------------------------------------
            case STATE_STARTUP: TX_SEND = TX_SEND_OFF;          // Make sure the transmitter is off.
                                NOP();
                                TX_RELAY = TX_RELAY_OFF;
                                NOP();
                                ON_AIR_LED = LED_OFF;
                                NOP();
                                ALARM_LED = LED_OFF;

                                GIE = 0;                        // Disable interrupts.
                                counter = 0;                    // Reset counter.
                                GIE = 1;                        // Enable interrupts.
                                while (counter < STARTUP_TIME); // Wait the required time to allow all voltages to set.

                                if (SW_STBY == SW_STBY_TX)      // Check standby switch.
                                {
                                    ON_AIR_LED = LED_ON;
                                    GIE = 0;                    // Disable interrupts.
                                    counter = 0;                // Reset counter.
                                    GIE = 1;                    // Enable interrupts.
                                    while (SW_STBY == SW_STBY_TX)
                                    {
                                        if (counter > LED_BLINK_TIME)
                                        {
                                            GIE = 0;            // Disable interrupts.
                                            counter = 0;        // Reset counter.
                                            GIE = 1;            // Enable interrupts.
                                            ON_AIR_LED = !ON_AIR_LED;   // Invert LED status.
                                        }
                                    }
                                }
                                ON_AIR_LED = LED_OFF;
                                state = STATE_STBY;
                                break;

// --------------------------------------------------------------------------
// In the "standby" state the transmitter is off air and the antenna is
// connected to the receiver. All the outputs are clear.
// As long as the "standby" switch is in the "standby" position, the state
// will not change preventing any transmission. If the "standby" switch is
// switched to the "transmit" position, the state will change according to
// the status of the "break-in" switch: if it's in the "manual" position,
// the state will change straight away, if it's in the "break-in" position,
// the state will change only when the CW key will be pressed. In order to
// switch the transmitter on, the next state is "transmit switch".
// If an alarm is detected, the state will change to the corresponding
// alarm state to show the user which alarm occurred.
// --------------------------------------------------------------------------
            case STATE_STBY:    TX_SEND = TX_SEND_OFF;          // Make sure the transmitter is off.
                                NOP();
                                TX_RELAY = TX_RELAY_OFF;
                                NOP();
                                ON_AIR_LED = LED_OFF;
                                NOP();
                                ALARM_LED = LED_OFF;
                                while (1)                       // And check the status of the control lines.
                                {
                                    if (ALARM_1 == ALARM_ON)
                                    {
                                        state = STATE_ALARM_1;
                                        break;
                                    }
                                    if (ALARM_2 == ALARM_ON)
                                    {
                                        state = STATE_ALARM_2;
                                        break;
                                    }
                                    if (SW_STBY == SW_STBY_TX)
                                    {
                                        if (SW_BKIN == SW_BKIN_MAN)
                                        {
                                            state = STATE_SWTX;
                                            break;
                                        }
                                        else if (CW_KEY == CW_KEY_DOWN)
                                        {
                                            state = STATE_SWTX;
                                            break;
                                        }
                                    }
                                }
                                break;

// --------------------------------------------------------------------------
// In the "transmit switch" state the antenna relay is switched on to
// initiate the transmit procedure. The MOSFETs are not enabled yet even if
// the CW key is down. In order to prevent transmitter damages, it important
// to wait enough time to allow the antenna relay to complete its motion and
// make sure the antenna is properly connected to the transmitter output.
// This time is in the 10ms range and is specified in the RELAY_ON_DELAY
// constant. When this time has elapsed, the state changes to "transmit" and
// the transmission can start.
// If the "standby" switch is switched back to the "standby" position the
// delay loop is immediately broken and the state changes to "receive
// switch", which is a mandatory step in order to come back to the
// "standby" state.
// If an alarm is detected, the state will change immediately to the
// corresponding alarm state.
// --------------------------------------------------------------------------
            case STATE_SWTX:    TX_SEND = TX_SEND_OFF;          // Switch transmitter relay on.
                                NOP();
                                TX_RELAY = TX_RELAY_ON;
                                NOP();
                                ON_AIR_LED = LED_OFF;
                                NOP();
                                ALARM_LED = LED_OFF;
                                GIE = 0;                        // Disable interrupts.
                                counter = 0;                    // Reset counter.
                                GIE = 1;                        // Enable interrupts.
                                state = STATE_TX;               // Prepare next state.
                                while (counter < RELAY_ON_DELAY)    // Wait the required time to allow the relay to complete its motion.
                                {
                                    if (ALARM_1 == ALARM_ON)
                                    {
                                        state = STATE_ALARM_1;
                                        break;
                                    }
                                    if (ALARM_2 == ALARM_ON)
                                    {
                                        state = STATE_ALARM_2;
                                        break;
                                    }
                                    if (SW_STBY == SW_STBY_STBY)
                                    {
                                        state = STATE_SWRX;
                                        break;
                                    }
                                }
                                break;
                                
// --------------------------------------------------------------------------
// In the "transmit" state the antenna relay has reliably connected the
// transmitter output to the antenna cable and the transmitter is ready to
// operate, but the MOSFETs are not enabled yet.
// The status of the CW key is monitored and as soon as it's pressed the
// state will change to "on air" where the transmission will occur.
// If the "standby" switch is switched back to the "standby" position the
// state will change to "receive switch" in order to terminate the
// transmission and switch back to the "standby" state.
// If the "break-in" switch is in the "break-in" position, the transmission
// will be automatically terminated after the time specified in the
// BREAK_IN_TIME has elapsed also by changing the state to "receive switch".
// If an alarm is detected, the state will change immediately to the
// corresponding alarm state.
// --------------------------------------------------------------------------
            case STATE_TX:      TX_SEND = TX_SEND_OFF;          // Transmitter relay on, but still off air.
                                NOP();
                                TX_RELAY = TX_RELAY_ON;
                                NOP();
                                ON_AIR_LED = LED_OFF;
                                NOP();
                                ALARM_LED = LED_OFF;
                                GIE = 0;                        // Disable interrupts.
                                counter = 0;                    // Reset counter.
                                GIE = 1;                        // Enable interrupts.
                                while (1)
                                {
                                    if (CW_KEY == CW_KEY_DOWN)
                                    {
                                        state = STATE_TX_ON_AIR;
                                        break;
                                    }    
                                    if (ALARM_1 == ALARM_ON)
                                    {
                                        state = STATE_ALARM_1;
                                        break;
                                    }
                                    if (ALARM_2 == ALARM_ON)
                                    {
                                        state = STATE_ALARM_2;
                                        break;
                                    }
                                    if (SW_STBY == SW_STBY_STBY)
                                    {
                                        state = STATE_SWRX;
                                        break;
                                    }
                                    if (((SW_BKIN == SW_BKIN_BKIN)) && (counter > BREAK_IN_TIME))
                                    {
                                        state = STATE_SWRX;
                                        break;
                                    }
                                }
                                break;

// --------------------------------------------------------------------------
// In the "on-air" state the RF is fed to the MOSFETs and the transmission
// takes place. In order to prevent transmitter damages, this state should
// only be selected by the "transmit" state that makes sure the antenna
// relay has reliably connected the transmitter output to the antenna cable.
// The status of the CW key is monitored and as soon as it's released the
// state will change back to "transmit", ready for the next key press.
// If the "standby" switch is switched back to the "standby" position the
// state will change to "receive switch" in order to terminate the
// transmission and switch back to the "standby" state.
// If an alarm is detected, the state will change immediately to the
// corresponding alarm state.
// --------------------------------------------------------------------------
            case STATE_TX_ON_AIR:TX_SEND = TX_SEND_ON;          // Now we are really transmitting.
                                NOP();
                                TX_RELAY = TX_RELAY_ON;
                                NOP();
                                ON_AIR_LED = LED_ON;
                                NOP();
                                ALARM_LED = LED_OFF;
                                while (1)
                                {
                                    if (CW_KEY == CW_KEY_UP)
                                    {
                                        state = STATE_TX;
                                        break;
                                    }    
                                    if (ALARM_1 == ALARM_ON)
                                    {
                                        state = STATE_ALARM_1;
                                        break;
                                    }
                                    if (ALARM_2 == ALARM_ON)
                                    {
                                        state = STATE_ALARM_2;
                                        break;
                                    }
                                    if (SW_STBY == SW_STBY_STBY)
                                    {
                                        state = STATE_SWRX;
                                        break;
                                    }
                                }    
                                break;

// --------------------------------------------------------------------------
// In the "receive switch" the MOSFETs have already been switched off and
// it's now time to de-energize the antenna relay. The RF power is
// immediately cut, but the relay needs some time to complete its motion and
// this state is intended to wait this amount of time, which is specified in
// the RELAY_OFF_DELAY constant, before switching back to the "standby"
// state.
// It's important to wait this time even when switching the relay off. This
// is to avoid that if the relay is immediately switched back on, the time 
// may not suffice to complete the motion.
// In this state the alarms are not checked because the transmitter is
// already off and they are checked in the "standby" state anyway.
// --------------------------------------------------------------------------
            case STATE_SWRX:    TX_SEND = TX_SEND_OFF;          // Switch transmitter off.
                                NOP();
                                TX_RELAY = TX_RELAY_OFF;
                                NOP();
                                ON_AIR_LED = LED_OFF;
                                NOP();
                                ALARM_LED = LED_OFF;
                                GIE = 0;                        // Disable interrupts.
                                counter = 0;                    // Reset counter.
                                GIE = 1;                        // Enable interrupts.
                                state = STATE_STBY;
                                while (counter < RELAY_OFF_DELAY);
                                break;

// --------------------------------------------------------------------------
// In the "alarm 1" state, the transmitter is immediately switched off and
// the alarm LED is switched on to show this state. In order to quit this
// state and go back to the "standby" state, the "standby" switch has to be
// switched back in the "standby" position and the alarm condition must
// disappear.
// --------------------------------------------------------------------------
            case STATE_ALARM_1: TX_SEND = TX_SEND_OFF;          // Alarm condition: switch off transmitter immediately.
                                NOP();
                                TX_RELAY = TX_RELAY_OFF;
                                NOP();
                                ON_AIR_LED = LED_OFF;
                                NOP();
                                ALARM_LED = LED_ON;
                                while (1)                       // Stay in alarm state as long as the standby switch is in "transmit" and the alarm is not cleared.
                                {
                                    if ((SW_STBY == SW_STBY_STBY) && (ALARM_1 == ALARM_OFF))
                                        break;
                                }
                                state = STATE_STBY;             // Reset alarm state and go back in standby mode.
                                break;

// --------------------------------------------------------------------------
// In the "alarm 2" state, the transmitter is immediately switched off and
// the alarm LED blinks to show this state. The blinking half period is
// specified in the LED_BLINK_TIME constant. In order to quit this state and
// go back to the "standby" state, the "standby" switch has to be switched
// back in the "standby" position and the alarm condition must disappear.
// --------------------------------------------------------------------------
            case STATE_ALARM_2: TX_SEND = TX_SEND_OFF;          // Alarm condition: switch off transmitter immediately.
                                NOP();
                                TX_RELAY = TX_RELAY_OFF;
                                NOP();
                                ON_AIR_LED = LED_OFF;
                                NOP();
                                ALARM_LED = LED_ON;
                                GIE = 0;                        // Disable interrupts.
                                counter = 0;                    // Reset counter.
                                GIE = 1;                        // Enable interrupts.
                                while (1)                       // Stay in alarm state as long as the standby switch is in "transmit" and the alarm is not cleared.
                                {
                                    if ((SW_STBY == SW_STBY_STBY) && (ALARM_2 == ALARM_OFF))
                                        break;
                                    if (counter > LED_BLINK_TIME)
                                    {
                                        GIE = 0;                // Disable interrupts.
                                        counter = 0;            // Reset counter.
                                        GIE = 1;                // Enable interrupts.
                                        ALARM_LED = !ALARM_LED; // Invert LED status.
                                    }
                                }
                                state = STATE_STBY;             // Reset alarm state and go back in standby mode.
                                break;

// --------------------------------------------------------------------------
// No other state should ever be possible but, just in case, the following
// state allows recovery in case of a software crash.
// --------------------------------------------------------------------------
            default:            state = STATE_STBY;             // Just in case of a software crash.
                                break;
        }
    }
}



